package com.vmware.samples.chassisrackvsphereui {

import com.vmware.actionsfw.ActionContext;
import com.vmware.actionsfw.events.ActionInvocationEvent;
import com.vmware.core.events.DataRefreshInvocationEvent;
import com.vmware.core.model.IResourceReference;
import com.vmware.core.model.StringDataObject;
import com.vmware.data.common.OperationType;
import com.vmware.data.common.events.ModelChangeEvent;
import com.vmware.data.query.events.PropertyRequest;
import com.vmware.flexutil.events.MethodReturnEvent;
import com.vmware.samples.chassisrackvsphereui.model.ChassisInfo;

import flash.events.EventDispatcher;

import mx.controls.Alert;

[Event(name="{com.vmware.data.common.events.ModelChangeEvent.MODEL_CHANGE}",
      type="com.vmware.data.common.events.ModelChangeEvent")]
[Event(name="{com.vmware.data.query.events.PropertyRequest.REQUEST_ID}",
      type="com.vmware.data.query.events.PropertyRequest")]
[Event(name="{com.vmware.core.events.DataRefreshInvocationEvent.EVENT_ID}",
       type="com.vmware.core.events.DataRefreshInvocationEvent")]
/**
 * Command class to handle the chassis actions defined in plugin.xml.
 *
 * Implementation notes:
 * - A [RequestHandler] method is defined for each ActionSpec uid.
 * - Actions that trigger a server call use a common ChassisServiceProxy instance.
 * - Proxy calls must handle the server's callbacks in the same class.
 * - ModelChangeEvent notifies the plaftorm when an object is added, changed or deleted.
 * - [EventHandler] methods should NOT be added here.
 */
public class ChassisActionCommand extends EventDispatcher {
   // Define a global proxy instance with the InjectableProxy tag
   [InjectableProxy]
   public var proxy:ChassisServiceProxy;

   // Index used for new chassis names.
   private static var _index:int = 1;

   [RequestHandler("com.vmware.samples.chassis.deleteChassis")]
   /**
    * Handles the delete chassis action invocation. This will remove the chassis from the
    * object store on the server after the user confirms.
    */
   public function onDeleteChassis(event:ActionInvocationEvent):void {
      const chassisRef:IResourceReference = getTargetObject(event);

      // Quick server call to retrieve the chassis name first.
      dispatchEvent(PropertyRequest.newInstance(chassisRef, "name"));
   }

   [ResponseHandler(name="{com.vmware.data.query.events.PropertyRequest.RESPONSE_ID}")]
   public function onChassisNameRetrieved(request:PropertyRequest,
                                          result:StringDataObject, error:Error):void {
      if ((error != null) || (result == null)) {
         // For instance an ObjectNotFoundError may occur if another user deleted that chassis
         var errorMsg:String = (error != null ?
               error.message : "null result getting chassis name");
         Alert.show(errorMsg);
         return;
      }

      const chassisRef:IResourceReference = request.providerObject;
      const chassisName:String = result.value;
      const context:ChassisActionContext = 
            new ChassisActionContext(chassisName, chassisRef, OperationType.DELETE);
      // chassisName could be used here to prompt the user to confirm the delete action

      proxy.deleteChassis(chassisRef, onDeleteChassisComplete, context);
   }

   [RequestHandler("com.vmware.samples.chassis.createChassis")]
   /**
    * Handles the Add Chassis action.
    */
   public function onCreateChassis(event:ActionInvocationEvent):void {
      // Use hardcoded data for simplicity, see chassis-ui project for a wizard example
      var chassisInfo:ChassisInfo = new ChassisInfo();
      chassisInfo.name = "Chassis +" + _index++;
      chassisInfo.dimensions = "25in x 32in x 18in";
      chassisInfo.serverType = "Server_Type 4";

     var targetRef:IResourceReference = null;
     if (isContextValid(event.context)) {
       // There is a valid context => Create chassis in this context
        targetRef = getTargetObject(event);
     }

      const context:ChassisActionContext =
            new ChassisActionContext(chassisInfo.name, null, OperationType.ADD);

      proxy.createChassis(chassisInfo, targetRef, onCreateChassisComplete, context);
   }

   [RequestHandler("com.vmware.samples.chassis.editChassis")]
   /**
    * Handles the Edit Chassis action. This triggers a simple change of the
    * chassis type property to simulate some editing action.
    */
   public function onEditChassis(event:ActionInvocationEvent):void {
      var chassisRef:IResourceReference = getTargetObject(event);
      const context:ChassisActionContext = new ChassisActionContext(
            null, chassisRef, OperationType.CHANGE);

      proxy.editChassis(chassisRef, onEditChassisComplete, context);
   }

   private function onCreateChassisComplete(event:MethodReturnEvent):void {
      if (event.error != null) {
         Alert.show(event.error.message);
         return;
      }

      // Another user may have already created a chassis with the same name
      const chassisCreated:Boolean = (event.result != null);
      const name:String = (event.callContext as ChassisActionContext).name;
      if (chassisCreated) {
         (event.callContext as ChassisActionContext).reference =
            event.result as IResourceReference;
      }
      const msg:String = Util.getString(
         (chassisCreated ? "createAction.success" : "createAction.error"), [name]);
      Alert.show(msg);

      refreshUi(event.callContext as ChassisActionContext);
   }

   private function onDeleteChassisComplete(event:MethodReturnEvent):void {
      if (event.error != null) {
         Alert.show(event.error.message);
         return;
      }

      if (!event.result) {
         // Another user must have deleted this chassis already
         const name:String = (event.callContext as ChassisActionContext).name;
         const msg:String = Util.getString("deleteAction.error", [name]);
         Alert.show(msg);
      }
      refreshUi(event.callContext as ChassisActionContext);
   }

   private function onEditChassisComplete(event:MethodReturnEvent):void {
      if (event.error != null) {
         Alert.show(event.error.message);
         return;
      }

      if (!event.result) {
         // Another user must have deleted this chassis already
         const msg:String = Util.getString("editAction.error");
         Alert.show(msg);
      }
      refreshUi(event.callContext as ChassisActionContext);
   }

   private function refreshUi(context:ChassisActionContext):void {
      if (context.reference != null) {
         // Send a ModelChangeEvent with the modified object to update the UI
         dispatchEvent(
            ModelChangeEvent.newSingleObjectChangeEvent(context.reference, context.opType));
      } else {
         // Use a global refresh event when we don't know the modified object,
         // in case a new chassis with the same name already exists.
         dispatchEvent(new DataRefreshInvocationEvent(DataRefreshInvocationEvent.EVENT_ID));
      }
   }

   /**
    * Get the action target as a chassis reference
    *
    * @param event the ActionInvocationEvent
    * @return the chassis on which the action was called.
    */
   private function getTargetObject(event:ActionInvocationEvent):IResourceReference {
      // Throw an error if actionContext is not valid
      var actionContext:ActionContext = event.context;
      if (!isContextValid(actionContext)) {
         throw (new Error( "Invalid action context for " + event.type));
      }
      return (actionContext.targetObjects[0] as IResourceReference);
   }

   /**
   * Checks whether the action context is avalable and valid.
   *
   * @param actionContext the action context to check
   * @return true if valid context is present, false otherwise.
   */
   private function isContextValid(actionContext:ActionContext):Boolean {
      // Consider actionContext valid if actionContext.targetObjects is an array with 1 object
      // (ActionSpec property acceptsMultipleTargets is false by default)
      return (actionContext != null) && (actionContext.targetObjects != null)
        && (actionContext.targetObjects.length > 0)
        && (actionContext.targetObjects[0] is IResourceReference);
   }
}
}

import com.vmware.core.model.IResourceReference;
import com.vmware.data.common.OperationType;

/**
 * Local helper class to hold context info
 */
class ChassisActionContext {
   public var name:String;
   public var reference:IResourceReference;
   public var opType:OperationType;

   public function ChassisActionContext(
      name:String, reference:IResourceReference, opType:OperationType) {
      this.name = name;
      this.reference = reference;
      this.opType = opType;
   }
}
